#!/usr/bin/env python3
#
# mfw lets you export and import Meraki MX appliance L3 outbound firewall rules
# and allows the use of objects and objects groups for addresses and services.
# The object and service names are embedded into the description field of the firewall
# rule in the dashboard allowing previsouly imported rules to be exported again, changed
# and re-imported.
#
# Installation:
# mfw uses dotenv to safely store your credentials.  Create a file called .meraki.env
# in your home directory.  For Linux this is typically /home/username.  For Windows this
# is typically
# c:\users\<username>.
# Into .meraki.env put this line:
# x_cisco_meraki_api_key=<your API key>
# If you don't have an API key yet then follow the instructions on this page:
# https://documentation.meraki.com/zGeneral_Administration/Other_Topics/The_Cisco_Meraki_Dashboard_API
#
# Prior to running this script you'll need Python 3.x installed and you'll need to run the below
# commands to install the extra components required.
# pip3 install argparse
# pip3 install meraki==1.0.0b3
# pip3 install -U meraki
# pip3 install -U python-dotenv
#
# If you are using the script on Linux I would suggest marking it executable to make running
# it simpler.
# chmod +x mfw.py
#
# Usage:
# To export the current rules (or a set of previously imported rules):
# mfw.py  -o "Your org name" -n "Your network name" export rules.csv objects.csv
# To import rules and append them to what is already in the dashboard:
# mfw.py  -o "Your org name" -n "Your network name" import -a rules.csv objects.csv
# To import rules and replace what is already in the dashboard:
# mfw.py  -o "Your org name" -n "Your network name" import -r rules.csv objects.csv
#
# objects.csv and rules.csv are - csv files.  They have been designed so you can edit them in
# Microsoft Excel.
# ojects.csv contains objects, object groups (which are other objects) and service defintions.  When
# they are imported a simple string substition is done for the object name with the values in the next
# column.  A sample object.csv might look like:
# Name,Value
# google,"google.com,accounts"
# web,"80,443"
# accounts,accounts.google.com
# In the above example, "google" is a group consisting of the FQDN google.com and the nested object
# called "accounts", which itself is another FQDN - accounts.google.com.
# "web" is a service group consisting of the ports 80 and 443.
#
# rules.csv might look like:
# Comment,Policy,Protocol,Source Port,Source CIDR,Destination Port,Destination CIDR,Syslog Enabled
# General Comment,allow,udp,Any,192.168.73.1/32,web,google,False
# ,deny,tcp,Any,192.168.73.2/32,80,google.com,False
# You'll notice the rule referencing the object group "google" and the service group "web".
#
# The script does not do any checking on the format of the files, so if you get an error from
# the meraki API during import you have put something in a format that the API does not like.
# You could try creating some of the rules in the dashboard and exporting them to see how they should
# look.
# You could try breaking up the rules you are importing into smaller files until you locate the rule
# with the error.
# Common things to look out for are to make sure all addresses are in prefix notation.  You can't just
# have an IP address on its own.  Also the second field in the objects.csv file MUST be in speech or
# quote marks.
#

import os,argparse,meraki,csv


# Load global and local Meraki settings such as x_cisco_meraki_api_key
from dotenv import load_dotenv
load_dotenv()
load_dotenv(dotenv_path=os.path.join(os.path.expanduser("~"),".meraki.env"))

dashboard = meraki.DashboardAPI(
	api_key=os.getenv("x_cisco_meraki_api_key"),
	base_url='https://api-mp.meraki.com/api/v1/',
	output_log=False,
	print_console=False
)



# This function retrieves the netId
def getNetId(orgName,netName):
	orgId=None
	netId=None

	# Search for the org
	for org in dashboard.organizations.getOrganizations():
		if org['name'] == orgName:
			orgId=org['id']
			break;

	if orgId == None:
		print("Invalid organization name supplied: "+orgName)			
		exit(-1)

	# Search for the network
	for net in dashboard.organizations.getOrganizationNetworks(orgId):
		if net['name'] == netName:
			netId=net['id']
			break;

	if netId == None:
		print("Invalid network name supplied: "+netName)			
		exit(-1)

	return netId


# This function attempts to load the existing firewall rules (which may not exist)
def load(netId):
	firewallRules={}
	objects={}

	firewallRules=dashboard.appliance.getNetworkApplianceFirewallL3FirewallRules(netId)["rules"]
	firewallRules.pop(); # Remove default rule

	parser = argparse.ArgumentParser()
	parser.add_argument("--srcPort", help="Source port")
	parser.add_argument("--srcCidr", help="Source address")
	parser.add_argument("--destPort", help="Destination port")
	parser.add_argument("--destCidr", help="Destination address")
	for rule in firewallRules:
		args,unknown=parser.parse_known_args(rule["comment"].split())
		
		if args.srcPort: objects[args.srcPort]=rule["srcPort"]
		if args.srcCidr: objects[args.srcCidr]=rule["srcCidr"]
		if args.destPort: objects[args.destPort]=rule["destPort"]
		if args.destCidr: objects[args.destCidr]=rule["destCidr"]

		rule["comment"]=' '.join(unknown)

	return(firewallRules,objects)

# Save the appliance layer 3 rules to local files
def exportFirewall(firewallRules,firewallObjects,rulesHandle,objectsHandle):
	csv_rules=csv.writer(rulesHandle, dialect='excel')
	csv_objects=csv.writer(objectsHandle, dialect='excel')

	csv_objects.writerow(["Name","Value"])
	for k, v in firewallObjects.items():
		csv_objects.writerow([k,v])

	csv_rules.writerow(["Comment", "Policy", "Protocol", "Source Port", "Source CIDR", "Destination Port", "Destination CIDR", "Syslog Enabled"])
	for rule in firewallRules:
		csv_rules.writerow(rule.values())


# Load the firewall rules from local files and apply them to the Meraki Dashboard network
def importFirewall(netId,firewallRules,firewallObjects,rulesHandle,objectsHandle):
	csv_objects=csv.reader(objectsHandle, dialect='excel')
	csv_rules=csv.reader(rulesHandle, dialect='excel')

	# Read in all the objects
	next(csv_objects) # Skip the header
	for row in csv_objects:
		# Only looks at rows containing exactly two values, and where the object and value are not the same
		if len(row)==2 and (row[0] != row[1]):
			firewallObjects[row[0]]=row[1]

	# Loop while checking if any object is a reference to another object
	while(True):
		changed=False
	
		for k, v in firewallObjects.items():
			newValue=""
			for row in v.split(","):
				if row in firewallObjects:
					newValue+=firewallObjects[row]+","
					changed=True;
				else:
					newValue+=row+","
			firewallObjects[k]=newValue[:-1];
		
		# If no object referenced any other object we can stop
		if(not changed):
			break;

	# Now finally ready in all the rules
	next(csv_rules) # Skip the header
	for row in csv_rules:
		if len(row)==8:
			for x in range(3,7):
				# Substitue any objects found
				if row[x] in firewallObjects:
					if x==3:
						row[0]+=" --srcPort "+row[x]
					elif x==4:
						row[0]+=" --srcCidr "+row[x]
					elif x==5:
						row[0]+=" --destPort "+row[x]
					elif x==6:
						row[0]+=" --destCidr "+row[x]

					row[x]=firewallObjects[row[x]]

			firewallRules.append({'comment':row[0],'policy':row[1],'protocol':row[2],'srcPort':row[3],'srcCidr':row[4],'destPort':row[5],'destCidr':row[6],'syslogEnabled':row[7]})

	# Apply the rules back to the dashboard
	dashboard.appliance.updateNetworkApplianceFirewallL3FirewallRules(netId,rules=firewallRules)


def main():
	# Meraki parameters
	orgName=None
	netName=None
	netId=None

	text="""
	mfw.py ...
	In your home diretory you should have a .meraki.env file containing x_cisco_meraki_api_key=<your API key>
	"""
	
	parser = argparse.ArgumentParser(description = text)
	parser.add_argument("-o", "--orgName", help="Meraki org name")
	parser.add_argument("-n", "--netName", help="Meraki network name")
	subparsers = parser.add_subparsers(dest="command", required=True, help='sub-command help')

	parser_a = subparsers.add_parser('export', help='Exports rules and objects to local files')
	parser_a.add_argument("rules", help="CSV file of access list rules",type=argparse.FileType('w'))
	parser_a.add_argument("objects", help="CSV file of objects",type=argparse.FileType('w'))

	parser_a = subparsers.add_parser('import', help='Imports rules and objects to the Meraki Dashboard from local files')
	group_a=parser_a.add_mutually_exclusive_group(required=True)
	group_a.add_argument("-a","--append", action='store_true', help="append the rules")
	group_a.add_argument("-r","--replace", action='store_true', help="replace the existing rules")
	parser_a.add_argument("rules", help="CSV file of access list rules",type=argparse.FileType('r'))
	parser_a.add_argument("objects", help="CSV file of objects",type=argparse.FileType('r'))

	args=parser.parse_args()
	
	orgName=os.getenv("orgName")
	netName=os.getenv("netName")

	if args.orgName: orgName=args.orgName
	if args.netName: netName=args.netName

	if not (os.getenv("x_cisco_meraki_api_key") or os.getenv("MERAKI_DASHBOARD_API_KEY")):
		print("x_cisco_meraki_api_key must be defined in .meraki.env in your home directory or in .env in the current directory")
		exit(-1)
	if not orgName:
		print("orgName must be defined on the command line, in .meraki.env in your home directory or in .env in the current directory")
		exit(-1)
	if not netName:
		print("netName must be defined on the command line, in .meraki.env in your home directory or in .env in the current directory")
		exit(-1)

	netId=getNetId(orgName,netName)

	firewallRules=[]
	firewallObjects={}
	if args.command == "export" or (args.command == "import" and args.append):
		print("* Loading existing firewall rules")
		(firewallRules,firewallObjects)=load(netId)

	if args.command == "import":
		print("* Importing firewall rules and objects and saving to Meraki Dashboard")
		importFirewall(netId,firewallRules,firewallObjects,args.rules,args.objects)
	elif args.command == "export":
		print("* Exporting firewall rules to local files")
		exportFirewall(firewallRules,firewallObjects,args.rules,args.objects)
		args.rules.close();
		args.objects.close();

main()
